/***************************************************************************
                          LOW_link.h  -  description
                             -------------------
    begin                : Sun Jul 7 2002
    copyright            : (C) 2002 by Harald Roelle, Helmut Reiser
    email                : roelle@informatik.uni-muenchen.de, reiser@informatik.uni-muenchen.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef LOW_LINK_H
#define LOW_LINK_H


#include "LOW_types.h"
#include "LOW_deviceIDRaw.h"
#include "LOW_exception.h"
#include "LOW_semaphoreSet.h"
#include "LOW_platformMisc.h"
#include "LOW_IPCKeyGenerator.h"


    
/** Abstract base class for 1-Wire link adapters.

    Any link class representing a concrete 1-Wire link adapter must inherit from this class.
    
    @author Harald Roelle, Helmut Reiser
 */
class LOW_link {

//=======================================================================================
public:
  
  //=====================================================================================
  //
  // exceptions
  //
 
  /** Exception base class for all exceptions thrown by LOW_link. */
  class_DERIVE_FROM_EXCEPTION( link_error, LOW_exception);

  class_DERIVE_FROM_EXCEPTION( comm_error,         link_error);
  class_DERIVE_FROM_EXCEPTION( internal_error,     link_error);
  class_DERIVE_FROM_EXCEPTION( illegalSpeed_error, link_error);
  class_DERIVE_FROM_EXCEPTION( illegalLevel_error, link_error);
  class_DERIVE_FROM_EXCEPTION( notAllowed_error,   link_error);
  class_DERIVE_FROM_EXCEPTION( sizeMismatch_error, link_error);

    
  //=====================================================================================
  //
  // locks
  //
  
  /** Locking class to ensure exclusive access to a link.

      The class is intended to be used in a "locking is creation" design pattern.
      On creation an exclusive lock is optained for the device, and on destruction
      the lock is released.
   */
  class commLock {
    public:
      /** Obtain the lock.

          @param inLink  Reference to the link the lock is for.
       */
      commLock( LOW_link &inLink);

      /** Release the lock.
          @throw internal_error  Thrown on illegal lock state (should not happen).
       */
      ~commLock();

    private:
      LOW_link  &link;  /**< Reference to the link the lock is for. */
  };

  
  //=====================================================================================
  //
  // type definitions
  //

  /** Vector type of link class pointers. */
  typedef std::vector<LOW_link*> linkPtrVec_t;
  
  /** Type for individual link ID number. */
  typedef uint32_t linkID_t;

  /** Type for strong pullup period specification. */
  typedef enum { pullUp_16_4=0x00,  /**<   16.4 ms */
                 pullUp_65_5,       /**<   65.5 ms */
                 pullUp_131,        /**<  131   ms */
                 pullUp_262,        /**<  262   ms */
                 pullUp_524,        /**<  524   ms */
                 pullUp_1048,       /**< 1048   ms */
                 pullUp_NONE=0xff   /**< no pullup */
               }  strongPullup_t;

    
  //=====================================================================================
  //
  // constructors
  //

  /** Destructor.
   */ 
  virtual ~LOW_link();

   
  //=====================================================================================
  //
  // operator overloading
  //
  
  bool operator==(LOW_link &inLink) const;  /**< Comparison based on linkID. */
  
 
  //=====================================================================================
  /**
      @name Bus touch (write/read) methods
   */
  //!@{
 
  /** Send 1 bit of communication to the 1-Wire net and return the
      result 1 bit read from the 1-Wire net.

      @param   inSendBit   Bit to send.
      @param   inPullup    Optional strong pullup time following the write/read cycle.
      
      @return  Bit that was reveived.
   */
  virtual bool touchBit( const bool inSendBit, const strongPullup_t inPullup = pullUp_NONE) = 0;
 
 
  /** Send 8 bits of communication to the 1-Wire net and return the
      result 8 bits read from the 1-Wire net. 

      @param   inSendByte  Byte to send.
      @param   inPullup    Optional strong pullup time following the 8 bit write/read cycle.
      
      @return  Byte that was reveived.
   */
  virtual uint8_t touchByte( const uint8_t inSendByte, const strongPullup_t inPullup = pullUp_NONE) = 0;
  
  
  /** Send block of communication to the 1-Wire net and return the
      resulting bytes read from the 1-Wire net. 

      <B>Note:</B>: When the strong pullup is selected it will appear after each byte
                    sent and NOT only after the last byte. 
      
      @param   inBytes     Byte block to send.
      @param   inPullup    Optional strong pullup time following each 8 bit write/read cycle.
      
      @return  Byte block that was reveived. Length is equal to number of sent bytes.
   */
  virtual byteVec_t touchBlock( const byteVec_t &inBytes, const strongPullup_t inPullup = pullUp_NONE) = 0;

  //!@}
  
  
  //=====================================================================================
  /**
      @name Bus read methods
   */
  //!@{
  
  /** Receive 1 bit from the 1-Wire net by previously sending one bit of 
      read communication to the 1-Wire net.

      @param   inPullup    Optional strong pullup time following the write/read cycle.
      
      @return  Bit that was reveived.
   */
  virtual bool readDataBit( const strongPullup_t inPullup = pullUp_NONE) = 0;

    
  /** Receive 1 byte from the 1-Wire net by previously sending
      8 bits of read communication to the 1-Wire net.

      @param   inPullup    Optional strong pullup time following the write/read cycle.
      
      @return  Byte that was reveived.
   */
  virtual uint8_t readDataByte( const strongPullup_t inPullup = pullUp_NONE) = 0;

    
  /** Receive a block of bytes from the 1-Wire net by previously sending
      a block of bytes of read communication to the 1-Wire Net.

      <B>Note:</B> When the strong pullup is selected it will appear after each byte
                   sent and NOT only after the last byte. 
      
      @param  outBytes   Values that were reveived. Read length is determined
                         by the preset length of the vector.
      
      @param  inPullup   Optional strong pullup time following each 8 bit write/read cycle.
   */
  virtual void readData( byteVec_t &outBytes, const strongPullup_t inPullup = pullUp_NONE) = 0;

  //!@}

    
  //=====================================================================================
  /**
      @name Bus write methods
   */
  //!@{
  
  /** Send 1 bit to the 1-Wire net and verify that the
      bit read from the 1-Wire net is the same (bus write operation).

      @param   inSendBit   Bit to send.
      @param   inPullup    Optional strong pullup time following the write/read cycle.
   */
  virtual void writeData( const bool inSendBit, const strongPullup_t inPullup = pullUp_NONE) = 0;

    
  /** Send 1 byte to the 1-Wire net and verify that the
      byte read from the 1-Wire net is the same (bus write operation).

      @param   inSendByte  Byte to send.
      @param   inPullup    Optional strong pullup time following the write/read cycle.
   */
  virtual void writeData( const uint8_t inSendByte, const strongPullup_t inPullup = pullUp_NONE) = 0;

    
  /** Send block of bytes to the 1-Wire net and verify that the
      bytes block read from the 1-Wire net are the same (bus write operation).

      <B>Note:</B>: When the strong pullup is selected it will appear after each byte
                    sent and NOT only after the last byte. 
      
      @param  inSendBytes  Block of bytes to send.
      @param  inPullup     Optional strong pullup time following each 8 bit write/read cycle.
   */
  virtual void writeData( const byteVec_t &inSendBytes, const strongPullup_t inPullup = pullUp_NONE) = 0;

  //!@}

    
  //=====================================================================================
  /**
      @name Misc methods
   */
  //!@{
  
  /** Get ID of the link.

      @return  ID of the link.
   */
  linkID_t getID() const;


  /** Get whether there is an external power line on the segment.

      @return  Whether there is an external power line on the segment.
  */
  bool getHasExternalPower() const;


  /** Reset the adapter.
      
      <B>Note:</B> This does not necessarily include a reset on the 1-Wire net.
                   Whether this is done or net is left to the concrete implementation. 
   */
  virtual void resetLinkAdapter() = 0;
  
  
  /** Reset all of the devices on the 1-Wire net.

      @return  true:  Presense pulse(s) detected, device(s) reset
               false: No presense pulses detected
   */
  virtual bool resetBus() = 0;
  
  
  /** Set the 1-Wire net line level to strong pullup for a specified time.
      
      @param    inMicroSecs    Pullup time in micro seconds.
   */
  virtual void strongPullup( const unsigned long inMicroSecs) = 0;
   
  
  /** Create a fixed 480 microseconds 12 volt pulse on the 1-Wire net 
      for programming EPROM devices.
  
      For EPROM programming, only a single slave device should be connected 
      to the 1-Wire bus and the cable must be short, not to exceed a few meters. 
      
      <B>Note:</B> One should not attempt generating a programming pulse with 
                   a non-EPROM device on the bus; this may damage the device 
                   as well as the link controller.
  
   */
  virtual void programPulse() = 0;


  /** Discover devices on the bus.

      @param  inBranchVector
      @param  outFoundID
      @param  outDiscrVec
   */  
  virtual void doSearchSequence( const LOW_deviceIDRaw &inBranchVector, 
                                 LOW_deviceIDRaw &outFoundID, LOW_deviceIDRaw &outDiscrVec) = 0;

  //!@}


            
//=======================================================================================
protected:

  //=====================================================================================
  //
  // static attributes
  //
  
  static linkID_t  linkCounter;  /**< Incremented on instance creation to get individual IDs. */

    
  //=====================================================================================
  //
  // type definitions
  //

  /** Semaphore numbers for semaphore set. */  
  typedef enum {
    counterSemNo = 0,
    lockSemNo,
    semaphoreCount    // MUST ALWAYS be the last!
  } semNum_t;

    
  //=====================================================================================
  //
  // attributes
  //
  
  const linkID_t    linkID;            /**< Individual ID of the link adapter. */
  bool              hasProgramPulse;   /**< Wether the adapter is capable of 12V Program pulse. */
  const bool        hasOverDrive;      /**< Wether the adapter is capable of overdrive bus speed. */
  const bool        hasExternalPower;  /**< Wether the attached bus supplies external power. */
  const bool        allowProgPulse;    /**< Wether the program pulse should be allowed. */
  LOW_semaphoreSet  *semSet;           /**< Semaphore set for locking. */

    
  //=====================================================================================
  //
  // constructors
  //
 
  /** Constructor.

      @param  inHasProgramPulse   Wether the adapter is capable of 12V Program pulse.
      @param  inHasOverDrive      Wether the adapter is capable of overdrive bus speed.
      @param  inHasExternalPower  Wether the attached bus supplies external power.
      @param  inAllowProgPulse    Wether the program pulse should be allowed.
   */
  LOW_link( const bool inHasProgramPulse, const bool inHasOverDrive, 
            const bool inHasExternalPower, const bool inAllowProgPulse = false);
    
  
//=======================================================================================
private:

  //=====================================================================================
  //
  // type definitions
  //
  
  friend class commLock;  /**< Needed to access the semaphore set. */
  
  
  //=====================================================================================
  //
  // attributes
  //
  
  unsigned int                             aquireCount;  /**< Counters how often a lock was aquired. */
  LOW_platformMiscFactory::threadIdent_t   aquirePID;    /**< Process which ows the lock. */
  
};
  

#endif
